/*
 * Decompiled with CFR 0.152.
 */
package org.geoserver.gwc.layer;

import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableSet;
import com.thoughtworks.xstream.XStream;
import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ForkJoinTask;
import java.util.concurrent.ForkJoinWorkerThread;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.geoserver.catalog.WorkspaceInfo;
import org.geoserver.config.AsynchResourceIterator;
import org.geoserver.config.util.SecureXStream;
import org.geoserver.gwc.layer.GeoServerTileLayerInfo;
import org.geoserver.gwc.layer.GeoServerTileLayerInfoImpl;
import org.geoserver.gwc.layer.TileLayerCatalog;
import org.geoserver.gwc.layer.TileLayerCatalogListener;
import org.geoserver.ows.LocalWorkspace;
import org.geoserver.platform.GeoServerResourceLoader;
import org.geoserver.platform.resource.Resource;
import org.geoserver.platform.resource.ResourceNotification;
import org.geoserver.platform.resource.Resources;
import org.geoserver.util.DimensionWarning;
import org.geotools.util.logging.Logging;
import org.geowebcache.config.ContextualConfigurationProvider;
import org.geowebcache.config.XMLConfiguration;
import org.geowebcache.storage.blobstore.file.FilePathUtils;

public class DefaultTileLayerCatalog
implements TileLayerCatalog {
    private static final Logger LOGGER = Logging.getLogger(DefaultTileLayerCatalog.class);
    private static final String LAYERINFO_DIRECTORY = "gwc-layers";
    private static final ForkJoinPool.ForkJoinWorkerThreadFactory INITIALLIZATION_THREAD_FACTORY = new ForkJoinPool.ForkJoinWorkerThreadFactory(){
        private AtomicInteger threadIdSeq = new AtomicInteger();

        @Override
        public ForkJoinWorkerThread newThread(ForkJoinPool pool) {
            String name = String.format("ForkJoinPool.%s-%d", DefaultTileLayerCatalog.class.getSimpleName(), this.threadIdSeq.incrementAndGet());
            ForkJoinWorkerThread thread = ForkJoinPool.defaultForkJoinWorkerThreadFactory.newThread(pool);
            thread.setName(name);
            return thread;
        }
    };
    private static final int INITIALIZATION_PARALLELISM = AsynchResourceIterator.ASYNCH_RESOURCE_THREADS;
    private static ThreadLocal<XStream> INITIALIZATION_SERIALIZER = new ThreadLocal();
    private ConcurrentMap<String, GeoServerTileLayerInfo> layersById;
    private Map<String, String> layersByName;
    private final Supplier<XStream> xstreamProvider;
    private final XStream serializer;
    private final GeoServerResourceLoader resourceLoader;
    private final String baseDirectory;
    private volatile boolean initialized;
    private List<TileLayerCatalogListener> listeners;

    public DefaultTileLayerCatalog(GeoServerResourceLoader resourceLoader, XMLConfiguration xmlPersisterFactory) throws IOException {
        this(resourceLoader, () -> xmlPersisterFactory.getConfiguredXStreamWithContext((XStream)new SecureXStream(), ContextualConfigurationProvider.Context.PERSIST));
    }

    DefaultTileLayerCatalog(GeoServerResourceLoader resourceLoader, Supplier<XStream> xstreamProvider) throws IOException {
        this.resourceLoader = resourceLoader;
        this.baseDirectory = LAYERINFO_DIRECTORY;
        this.layersByName = new ConcurrentHashMap<String, String>();
        this.layersById = new ConcurrentHashMap<String, GeoServerTileLayerInfo>();
        this.listeners = new ArrayList<TileLayerCatalogListener>();
        this.initialized = false;
        this.xstreamProvider = xstreamProvider;
        this.serializer = this.newXStream();
        resourceLoader.get(this.baseDirectory).addListener(this::handleBaseDirectoryResourceEvent);
    }

    private XStream newXStream() {
        XStream serializer = this.xstreamProvider.get();
        serializer.allowTypeHierarchy(GeoServerTileLayerInfo.class);
        serializer.allowTypes(new Class[]{DimensionWarning.WarningType.class});
        serializer.allowTypes(new String[]{"java.util.Collections$UnmodifiableSet"});
        serializer.addDefaultImplementation(LinkedHashSet.class, Set.class);
        serializer.alias("warning", DimensionWarning.WarningType.class);
        return serializer;
    }

    private void handleBaseDirectoryResourceEvent(ResourceNotification notify) {
        notify.events().forEach(this::handleBaseDirectoryResourceEvent);
    }

    private void handleBaseDirectoryResourceEvent(ResourceNotification.Event event) {
        TileLayerCatalogListener.Type tileEventType;
        GeoServerTileLayerInfoImpl layerInfo;
        boolean isLayerFile;
        String path = event.getPath();
        boolean bl = isLayerFile = !path.contains("/") && path.toLowerCase().endsWith(".xml");
        if (!isLayerFile) {
            return;
        }
        if (event.getKind() == ResourceNotification.Kind.ENTRY_DELETE) {
            String layerIdName = this.layersById.keySet().parallelStream().map(this::layerIdToFileName).filter(path::equals).findFirst().orElse(null);
            if (layerIdName == null) {
                return;
            }
            Preconditions.checkState((boolean)layerIdName.endsWith(".xml"));
            String layerId = layerIdName.substring(0, layerIdName.lastIndexOf(".xml"));
            GeoServerTileLayerInfo removed = (GeoServerTileLayerInfo)this.layersById.remove(layerId);
            if (removed != null) {
                this.layersByName.remove(removed.getName());
            }
            this.listeners.forEach(l -> l.onEvent(layerId, TileLayerCatalogListener.Type.DELETE));
            return;
        }
        Resource resource = this.resourceLoader.get(this.baseDirectory).get(path);
        try {
            layerInfo = this.depersist(resource);
        }
        catch (IOException e) {
            LOGGER.log(Level.SEVERE, "Error depersisting tile layer information from file " + resource.name(), e);
            return;
        }
        String layerId = layerInfo.getId();
        GeoServerTileLayerInfo currentInfo = (GeoServerTileLayerInfo)this.layersById.get(layerId);
        TileLayerCatalogListener.Type type = tileEventType = event.getKind() == ResourceNotification.Kind.ENTRY_CREATE ? TileLayerCatalogListener.Type.CREATE : TileLayerCatalogListener.Type.MODIFY;
        if (event.getKind() == ResourceNotification.Kind.ENTRY_MODIFY && currentInfo != null && !currentInfo.getName().contentEquals(layerInfo.getName())) {
            this.layersByName.remove(currentInfo.getName());
        }
        this.saveInternal(layerInfo);
        this.listeners.forEach(l -> l.onEvent(layerId, tileEventType));
    }

    @Override
    public void reset() {
        this.layersById.clear();
        this.layersByName.clear();
        this.initialized = false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void initialize() {
        this.reset();
        Resource baseDir = this.resourceLoader.get(this.baseDirectory);
        LOGGER.config("GeoServer TileLayer store base directory is: " + baseDir.path());
        LOGGER.config("Loading tile layers from " + baseDir.path());
        Stopwatch sw = Stopwatch.createStarted();
        Resources.ExtensionFilter xmlFilter = new Resources.ExtensionFilter(new String[]{"XML"});
        ForkJoinPool pool = new ForkJoinPool(INITIALIZATION_PARALLELISM, INITIALLIZATION_THREAD_FACTORY, null, false);
        try {
            ((ForkJoinTask)pool.submit(() -> baseDir.list().parallelStream().filter(r -> xmlFilter.accept(r)).forEach(this::initializationLoad))).join();
        }
        finally {
            pool.shutdownNow();
        }
        LOGGER.config(String.format("Loaded %,d tile layers in %s", this.layersById.size(), sw.stop()));
        this.initialized = true;
    }

    private GeoServerTileLayerInfoImpl initializationLoad(Resource res) {
        GeoServerTileLayerInfoImpl info;
        XStream unmarshaller = INITIALIZATION_SERIALIZER.get();
        if (unmarshaller == null) {
            unmarshaller = this.newXStream();
            INITIALIZATION_SERIALIZER.set(unmarshaller);
        }
        try {
            info = this.depersist(res, unmarshaller);
        }
        catch (Exception e) {
            LOGGER.log(Level.SEVERE, "Error depersisting tile layer information from file " + res.name(), e);
            return null;
        }
        this.saveInternal(info);
        if (LOGGER.isLoggable(Level.FINER)) {
            LOGGER.finer("Loaded tile layer '" + info.getName() + "'");
        }
        return info;
    }

    @Override
    public GeoServerTileLayerInfo getLayerById(String id) {
        this.checkInitialized();
        GeoServerTileLayerInfo layer = (GeoServerTileLayerInfo)this.layersById.get(id);
        return layer == null ? null : layer.clone();
    }

    private void checkInitialized() {
        Preconditions.checkState((boolean)this.initialized, (Object)"DefaultTileLayerCatalog is not initialized");
    }

    @Override
    public GeoServerTileLayerInfo getLayerByName(String layerName) {
        this.checkInitialized();
        String id = this.getLayerId(layerName);
        if (id == null) {
            return null;
        }
        return this.getLayerById(id);
    }

    @Override
    public Set<String> getLayerIds() {
        this.checkInitialized();
        return ImmutableSet.copyOf(this.layersById.keySet());
    }

    @Override
    public boolean exists(String layerId) {
        this.checkInitialized();
        return this.layersById.containsKey(layerId);
    }

    @Override
    public Set<String> getLayerNames() {
        this.checkInitialized();
        return ImmutableSet.copyOf(this.layersByName.keySet());
    }

    @Override
    public GeoServerTileLayerInfo delete(String tileLayerId) {
        this.checkInitialized();
        try {
            GeoServerTileLayerInfo currValue = (GeoServerTileLayerInfo)this.layersById.remove(tileLayerId);
            if (currValue != null) {
                Resource file = this.getFile(tileLayerId);
                this.layersByName.remove(currValue.getName());
                file.delete();
                this.listeners.forEach(l -> l.onEvent(tileLayerId, TileLayerCatalogListener.Type.DELETE));
                return currValue;
            }
        }
        catch (IOException notFound) {
            LOGGER.log(Level.FINEST, "Deleting " + tileLayerId, notFound);
        }
        return null;
    }

    @Override
    public GeoServerTileLayerInfo save(GeoServerTileLayerInfo newValue) {
        this.checkInitialized();
        GeoServerTileLayerInfoImpl oldValue = null;
        String tileLayerId = newValue.getId();
        Preconditions.checkNotNull((Object)tileLayerId);
        try {
            try {
                oldValue = this.loadInternal(tileLayerId);
            }
            catch (FileNotFoundException fileNotFoundException) {
            }
            catch (Exception other) {
                Throwables.throwIfUnchecked((Throwable)other);
            }
            if (oldValue == null) {
                String duplicateNameId = this.layersByName.get(newValue.getName());
                if (null != duplicateNameId) {
                    throw new IllegalArgumentException("TileLayer with same name already exists: " + newValue.getName() + ": <" + duplicateNameId + ">");
                }
            } else {
                this.layersByName.remove(oldValue.getName());
            }
            this.persist(newValue);
            this.layersByName.put(newValue.getName(), newValue.getId());
            this.layersById.put(newValue.getId(), newValue.clone());
        }
        catch (Exception e) {
            if (e instanceof ExecutionException) {
                Throwables.throwIfUnchecked((Throwable)e.getCause());
            }
            Throwables.throwIfUnchecked((Throwable)e);
        }
        return oldValue;
    }

    private void saveInternal(GeoServerTileLayerInfoImpl info) {
        this.layersByName.put(info.getName(), info.getId());
        this.layersById.put(info.getId(), info);
    }

    private void persist(GeoServerTileLayerInfo real) throws IOException {
        String tileLayerId = real.getId();
        Resource file = this.getFile(tileLayerId);
        boolean cleanup = false;
        if (file.getType() == Resource.Type.UNDEFINED) {
            cleanup = true;
        }
        Resource tmp = file.parent().get(file.name() + ".tmp");
        try (OutputStreamWriter writer = new OutputStreamWriter(tmp.out(), StandardCharsets.UTF_8);){
            this.serializer.toXML((Object)real, (Writer)writer);
        }
        catch (Exception e) {
            tmp.delete();
            if (cleanup) {
                file.delete();
            }
            Throwables.throwIfInstanceOf((Throwable)e, IOException.class);
            Throwables.throwIfUnchecked((Throwable)e);
        }
        try {
            this.depersist(tmp);
        }
        catch (Exception e) {
            LOGGER.log(Level.WARNING, "Persisted version of tile layer " + real.getName() + " can't be loaded back", e);
            Throwables.throwIfInstanceOf((Throwable)e, IOException.class);
            Throwables.throwIfUnchecked((Throwable)e);
            throw new RuntimeException(e);
        }
        this.rename(tmp, file);
    }

    private GeoServerTileLayerInfoImpl loadInternal(String tileLayerId) throws FileNotFoundException, IOException {
        Resource file = this.getFile(tileLayerId);
        if (file.getType() == Resource.Type.UNDEFINED) {
            throw new FileNotFoundException(tileLayerId);
        }
        return this.depersist(file);
    }

    private Resource getFile(String tileLayerId) throws IOException {
        String fileName = this.layerIdToFileName(tileLayerId);
        Resource base = this.resourceLoader.get(this.baseDirectory);
        return base.get(fileName);
    }

    private String layerIdToFileName(String tileLayerId) {
        return FilePathUtils.filteredLayerName((String)tileLayerId) + ".xml";
    }

    private GeoServerTileLayerInfoImpl depersist(Resource res) throws IOException {
        return this.depersist(res, this.serializer);
    }

    private GeoServerTileLayerInfoImpl depersist(Resource res, XStream unmarshaller) throws IOException {
        GeoServerTileLayerInfoImpl info;
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine("Depersisting GeoServerTileLayerInfo from " + res.path());
        }
        try (InputStreamReader reader = new InputStreamReader((InputStream)new ByteArrayInputStream(res.getContents()), StandardCharsets.UTF_8);){
            info = (GeoServerTileLayerInfoImpl)unmarshaller.fromXML((Reader)reader);
        }
        return info;
    }

    private void rename(Resource source, Resource dest) throws IOException {
        if (source.equals(dest)) {
            return;
        }
        boolean win = System.getProperty("os.name").startsWith("Windows");
        if (win && Resources.exists((Resource)dest)) {
            if (!dest.delete()) {
                throw new IOException("Could not delete: " + dest.path());
            }
            source.renameTo(dest);
        } else {
            source.renameTo(dest);
        }
    }

    @Override
    public String getLayerId(String layerName) {
        this.checkInitialized();
        WorkspaceInfo ws = LocalWorkspace.get();
        if (ws != null && !((String)layerName).startsWith(ws.getName() + ":")) {
            layerName = ws.getName() + ":" + (String)layerName;
        }
        return this.layersByName.get(layerName);
    }

    @Override
    public String getLayerName(String layerId) {
        this.checkInitialized();
        return ((GeoServerTileLayerInfo)this.layersById.get(layerId)).getName();
    }

    @Override
    public String getPersistenceLocation() {
        return this.resourceLoader.get(this.baseDirectory).path();
    }

    @Override
    public void addListener(TileLayerCatalogListener listener) {
        this.listeners.add(listener);
    }
}

