/*
 * Decompiled with CFR 0.152.
 */
package org.geotools.data.mongodb;

import com.mongodb.BasicDBObject;
import com.mongodb.DB;
import com.mongodb.DBCollection;
import com.mongodb.DBObject;
import com.mongodb.MongoClient;
import com.mongodb.MongoClientURI;
import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.StreamSupport;
import org.bson.BsonDocument;
import org.bson.BsonString;
import org.bson.BsonValue;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.geotools.data.FeatureWriter;
import org.geotools.data.Transaction;
import org.geotools.data.mongodb.MongoFeatureStore;
import org.geotools.data.mongodb.MongoSchemaDBStore;
import org.geotools.data.mongodb.MongoSchemaFileStore;
import org.geotools.data.mongodb.MongoSchemaInitParams;
import org.geotools.data.mongodb.MongoSchemaStore;
import org.geotools.data.mongodb.MongoUtil;
import org.geotools.data.mongodb.complex.JsonSelectAllFunction;
import org.geotools.data.mongodb.complex.JsonSelectFunction;
import org.geotools.data.mongodb.data.SchemaStoreDirectoryProvider;
import org.geotools.data.store.ContentDataStore;
import org.geotools.data.store.ContentEntry;
import org.geotools.data.store.ContentFeatureSource;
import org.geotools.data.store.ContentState;
import org.geotools.feature.simple.SimpleFeatureTypeBuilder;
import org.geotools.filter.FilterCapabilities;
import org.geotools.http.HTTPClient;
import org.geotools.referencing.CRS;
import org.geotools.referencing.crs.DefaultGeographicCRS;
import org.geotools.util.logging.Logging;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.feature.type.AttributeDescriptor;
import org.opengis.feature.type.Name;
import org.opengis.filter.And;
import org.opengis.filter.Filter;
import org.opengis.filter.Id;
import org.opengis.filter.Not;
import org.opengis.filter.PropertyIsBetween;
import org.opengis.filter.PropertyIsLike;
import org.opengis.filter.PropertyIsNull;
import org.opengis.filter.spatial.BBOX;
import org.opengis.filter.spatial.Intersects;
import org.opengis.filter.spatial.Within;
import org.opengis.referencing.crs.CoordinateReferenceSystem;

public class MongoDataStore
extends ContentDataStore {
    private static final Logger LOGGER = Logging.getLogger(MongoDataStore.class);
    static final String KEY_mapping = "mapping";
    static final String KEY_encoding = "encoding";
    static final String KEY_collection = "collection";
    final MongoSchemaStore schemaStore;
    final MongoClient dataStoreClient;
    final DB dataStoreDB;
    final boolean deactivateOrNativeFilter;
    private HTTPClient httpClient;
    FilterCapabilities filterCapabilities;
    private MongoSchemaInitParams schemaInitParams;

    public MongoDataStore(String dataStoreURI) {
        this(dataStoreURI, null);
    }

    public MongoDataStore(String dataStoreURI, String schemaStoreURI) {
        this(dataStoreURI, schemaStoreURI, true);
    }

    public MongoDataStore(String dataStoreURI, String schemaStoreURI, boolean createDatabaseIfNeeded) {
        this(dataStoreURI, schemaStoreURI, createDatabaseIfNeeded, null, null);
    }

    public MongoDataStore(String dataStoreURI, String schemaStoreURI, boolean createDatabaseIfNeeded, HTTPClient httpClient) {
        this(dataStoreURI, schemaStoreURI, createDatabaseIfNeeded, null, httpClient);
    }

    public MongoDataStore(String dataStoreURI, String schemaStoreURI, boolean createDatabaseIfNeeded, MongoSchemaInitParams schemaInitParams, HTTPClient httpClient) {
        MongoClientURI dataStoreClientURI = this.createMongoClientURI(dataStoreURI);
        this.dataStoreClient = this.createMongoClient(dataStoreClientURI);
        this.dataStoreDB = this.createDB(this.dataStoreClient, dataStoreClientURI.getDatabase(), !createDatabaseIfNeeded);
        if (this.dataStoreDB == null) {
            this.dataStoreClient.close();
            throw new IllegalArgumentException("Unknown mongodb database, \"" + dataStoreClientURI.getDatabase() + "\"");
        }
        this.deactivateOrNativeFilter = this.isMongoVersionLessThan2_6(dataStoreClientURI);
        this.httpClient = httpClient;
        this.schemaStore = this.createSchemaStore(schemaStoreURI);
        if (this.schemaStore == null) {
            this.dataStoreClient.close();
            throw new IllegalArgumentException("Unable to initialize schema store with URI \"" + schemaStoreURI + "\"");
        }
        this.filterCapabilities = this.createFilterCapabilties();
        this.schemaInitParams = schemaInitParams != null ? schemaInitParams : MongoSchemaInitParams.builder().build();
    }

    private boolean isMongoVersionLessThan2_6(MongoClientURI dataStoreClientURI) {
        boolean deactivateOrAux = false;
        if (this.dataStoreClient != null && dataStoreClientURI != null && dataStoreClientURI.getDatabase() != null) {
            List versionArray;
            Document result = this.dataStoreClient.getDatabase(dataStoreClientURI.getDatabase()).runCommand((Bson)new BsonDocument("buildinfo", (BsonValue)new BsonString("")));
            if (result.containsKey((Object)"versionArray") && ((Integer)(versionArray = (List)result.get((Object)"versionArray")).get(0) < 2 || (Integer)versionArray.get(0) == 2 && (Integer)versionArray.get(1) < 6)) {
                deactivateOrAux = true;
            }
        } else {
            throw new IllegalArgumentException("Unknown Mongo Client");
        }
        return deactivateOrAux;
    }

    final MongoClientURI createMongoClientURI(String dataStoreURI) {
        if (dataStoreURI == null) {
            throw new IllegalArgumentException("dataStoreURI may not be null");
        }
        if (!dataStoreURI.startsWith("mongodb://")) {
            throw new IllegalArgumentException("incorrect scheme for URI, expected to begin with \"mongodb://\", found URI of \"" + dataStoreURI + "\"");
        }
        return new MongoClientURI(dataStoreURI.toString());
    }

    final MongoClient createMongoClient(MongoClientURI mongoClientURI) {
        try {
            return new MongoClient(mongoClientURI);
        }
        catch (Exception e) {
            throw new IllegalArgumentException("Unknown mongodb host(s)", e);
        }
    }

    final DB createDB(MongoClient mongoClient, String databaseName, boolean databaseMustExist) {
        if (databaseMustExist && !StreamSupport.stream(mongoClient.listDatabaseNames().spliterator(), false).anyMatch(name -> databaseName.equalsIgnoreCase((String)name))) {
            return null;
        }
        return mongoClient.getDB(databaseName);
    }

    private synchronized MongoSchemaStore createSchemaStore(String schemaStoreURI) {
        if (schemaStoreURI.startsWith("file:")) {
            try {
                return new MongoSchemaFileStore(schemaStoreURI);
            }
            catch (IOException | URISyntaxException e) {
                LOGGER.log(Level.SEVERE, "Unable to create file-based schema store with URI \"" + schemaStoreURI + "\"", e);
            }
        } else if (schemaStoreURI.startsWith("mongodb:")) {
            try {
                return new MongoSchemaDBStore(schemaStoreURI);
            }
            catch (IOException e) {
                LOGGER.log(Level.SEVERE, "Unable to create mongodb-based schema store with URI \"" + schemaStoreURI + "\"", e);
            }
        } else if (schemaStoreURI.startsWith("http")) {
            try {
                File downloadedFile = MongoUtil.downloadSchemaFile(this.dataStoreDB.getName(), new URL(schemaStoreURI), this.httpClient, SchemaStoreDirectoryProvider.getHighestPriority());
                if (MongoUtil.isZipFile(downloadedFile)) {
                    File extractedFileLocation = MongoUtil.extractZipFile(downloadedFile.getParentFile(), downloadedFile);
                    LOGGER.log(Level.INFO, "Found Schema Files at " + extractedFileLocation.toString() + "after extracting ");
                    return new MongoSchemaFileStore(extractedFileLocation.toURI());
                }
                return new MongoSchemaFileStore(downloadedFile.getParentFile().toURI());
            }
            catch (IOException e) {
                LOGGER.log(Level.SEVERE, "Unable to create file-based schema store with URI \"" + schemaStoreURI + "\"", e);
            }
        } else {
            try {
                return new MongoSchemaFileStore("file:" + schemaStoreURI);
            }
            catch (IOException | URISyntaxException e) {
                LOGGER.log(Level.SEVERE, "Unable to create file-based schema store with URI \"" + schemaStoreURI + "\"", e);
            }
        }
        LOGGER.log(Level.SEVERE, "Unsupported URI \"{0}\" for schema store", schemaStoreURI);
        return null;
    }

    final FilterCapabilities createFilterCapabilties() {
        FilterCapabilities capabilities = new FilterCapabilities();
        if (this.deactivateOrNativeFilter) {
            capabilities.addType(And.class);
            capabilities.addType(Not.class);
        } else {
            capabilities.addAll(FilterCapabilities.LOGICAL_OPENGIS);
        }
        capabilities.addAll(FilterCapabilities.SIMPLE_COMPARISONS_OPENGIS);
        capabilities.addType(PropertyIsNull.class);
        capabilities.addType(PropertyIsBetween.class);
        capabilities.addType(PropertyIsLike.class);
        capabilities.addType(BBOX.class);
        capabilities.addType(Intersects.class);
        capabilities.addType(Within.class);
        capabilities.addType(Id.class);
        capabilities.addType(JsonSelectFunction.class);
        capabilities.addType(JsonSelectAllFunction.class);
        return capabilities;
    }

    public FilterCapabilities getFilterCapabilities() {
        return this.filterCapabilities;
    }

    public void createSchema(SimpleFeatureType incoming) throws IOException {
        String geometryMapping = "geometry";
        CoordinateReferenceSystem incomingCRS = incoming.getCoordinateReferenceSystem();
        if (incomingCRS == null) {
            incoming.getGeometryDescriptor().getCoordinateReferenceSystem();
        }
        if (!CRS.equalsIgnoreMetadata((Object)incomingCRS, (Object)DefaultGeographicCRS.WGS84)) {
            throw new IllegalArgumentException("Unsupported coordinate reference system, only WGS84 supported");
        }
        SimpleFeatureTypeBuilder builder = new SimpleFeatureTypeBuilder();
        builder.init(incoming);
        builder.setName(this.name(incoming.getTypeName()));
        incoming = builder.buildFeatureType();
        String gdName = incoming.getGeometryDescriptor().getLocalName();
        for (AttributeDescriptor ad : incoming.getAttributeDescriptors()) {
            String adName = ad.getLocalName();
            if (gdName.equals(adName)) {
                ad.getUserData().put(KEY_mapping, "geometry");
                ad.getUserData().put(KEY_encoding, "GeoJSON");
                continue;
            }
            ad.getUserData().put(KEY_mapping, "properties." + adName);
        }
        incoming.getUserData().put(KEY_collection, incoming.getTypeName());
        this.dataStoreDB.createCollection(incoming.getTypeName(), (DBObject)new BasicDBObject()).createIndex((DBObject)new BasicDBObject("geometry", (Object)"2dsphere"));
        ContentEntry entry = this.entry(incoming.getName());
        ContentState state = entry.getState(null);
        state.setFeatureType(incoming);
        this.schemaStore.storeSchema(incoming);
    }

    private String collectionNameFromType(SimpleFeatureType type) {
        String collectionName = (String)type.getUserData().get(KEY_collection);
        return collectionName != null ? collectionName : type.getTypeName();
    }

    protected List<Name> createTypeNames() throws IOException {
        LinkedHashSet collectionNames = new LinkedHashSet(this.dataStoreDB.getCollectionNames());
        LinkedHashSet<String> typeNameSet = new LinkedHashSet<String>();
        for (String string : this.getSchemaStore().typeNames()) {
            try {
                SimpleFeatureType candidateSchema = this.getSchemaStore().retrieveSchema(this.name(string));
                String string2 = this.collectionNameFromType(candidateSchema);
                if (collectionNames.contains(string2)) {
                    String geometryName = candidateSchema.getGeometryDescriptor().getLocalName();
                    String geometryMapping = (String)candidateSchema.getDescriptor(geometryName).getUserData().get(KEY_mapping);
                    if (geometryMapping != null) {
                        DBCollection collection = this.dataStoreDB.getCollection(string2);
                        Set<String> geometryIndices = MongoUtil.findIndexedGeometries(collection);
                        if (geometryIndices.contains(geometryMapping)) {
                            typeNameSet.add(string);
                            continue;
                        }
                        LOGGER.log(Level.WARNING, "Ignoring type \"{0}\", the geometry attribute, \"{1}\", is mapped to document key \"{2}\" but it is not spatialy indexed in collection {3}", new Object[]{this.name(string), geometryName, geometryMapping, collection.getFullName()});
                        continue;
                    }
                    LOGGER.log(Level.WARNING, "Ignoring type \"{0}\", the geometry attribute \"{1}\" is not mapped to a document key", new Object[]{this.name(string), geometryName});
                    continue;
                }
                LOGGER.log(Level.WARNING, "Ignoring type \"{0}\", the collection it maps \"{1}.{2}\" does not exist", new Object[]{this.name(string), this.dataStoreDB.getName(), string2});
            }
            catch (IOException e) {
                LOGGER.log(Level.WARNING, "Ignoring type \"{0}\", an exception was thrown while attempting to retrieve the schema: {1}", new Object[]{this.name(string), e});
            }
        }
        LinkedList collectionsToCheck = new LinkedList(collectionNames);
        collectionsToCheck.removeAll(typeNameSet);
        for (String collectionName : collectionsToCheck) {
            if (collectionName.startsWith("system.")) continue;
            DBCollection dBCollection = this.dataStoreDB.getCollection(collectionName);
            Set<String> geometryIndexSet = MongoUtil.findIndexedGeometries(dBCollection);
            if (!geometryIndexSet.isEmpty()) {
                typeNameSet.add(collectionName);
                continue;
            }
            LOGGER.log(Level.INFO, "Ignoring collection \"{0}\", unable to find key with spatial index", new Object[]{dBCollection.getFullName()});
        }
        ArrayList<Name> arrayList = new ArrayList<Name>();
        for (String string : typeNameSet) {
            arrayList.add(this.name(string));
        }
        return arrayList;
    }

    protected ContentFeatureSource createFeatureSource(ContentEntry entry) throws IOException {
        ContentState state = entry.getState(null);
        SimpleFeatureType type = state.getFeatureType();
        if (type == null && (type = this.schemaStore.retrieveSchema(entry.getName())) != null) {
            state.setFeatureType(type);
        }
        String collectionName = type != null ? this.collectionNameFromType(type) : entry.getTypeName();
        return new MongoFeatureStore(entry, null, this.dataStoreDB.getCollection(collectionName));
    }

    public FeatureWriter<SimpleFeatureType, SimpleFeature> getFeatureWriter(String typeName, Filter filter, Transaction tx) throws IOException {
        if (tx != Transaction.AUTO_COMMIT) {
            throw new IllegalArgumentException("Transactions not currently supported");
        }
        return super.getFeatureWriter(typeName, filter, tx);
    }

    protected ContentState createContentState(ContentEntry entry) {
        ContentState state = super.createContentState(entry);
        try {
            SimpleFeatureType type = this.schemaStore.retrieveSchema(entry.getName());
            if (type != null) {
                state.setFeatureType(type);
            }
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, "Exception thrown while attempting to retrieve the schema for {0}: {1}", new Object[]{entry.getName(), e});
        }
        return state;
    }

    public final MongoSchemaStore getSchemaStore() {
        return this.schemaStore;
    }

    public void dispose() {
        this.dataStoreClient.close();
        this.schemaStore.close();
        super.dispose();
    }

    public void cleanEntries() {
        LOGGER.info("Proceeding to clean all store cached entries");
        for (ContentEntry entry : this.entries.values()) {
            entry.dispose();
        }
        this.entries.clear();
    }

    public Optional<MongoSchemaInitParams> getSchemaInitParams() {
        return Optional.ofNullable(this.schemaInitParams);
    }

    public void setSchemaInitParams(MongoSchemaInitParams schemaInitParams) {
        this.schemaInitParams = schemaInitParams;
    }
}

