/*
 * Decompiled with CFR 0.152.
 */
package org.geotools.data.shapefile;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.geotools.data.shapefile.ShapefileIndexerBoundsHelper;
import org.geotools.data.shapefile.files.FileWriter;
import org.geotools.data.shapefile.files.ShpFileType;
import org.geotools.data.shapefile.files.ShpFiles;
import org.geotools.data.shapefile.files.StorageFile;
import org.geotools.data.shapefile.index.LockTimeoutException;
import org.geotools.data.shapefile.index.TreeException;
import org.geotools.data.shapefile.index.quadtree.Node;
import org.geotools.data.shapefile.index.quadtree.QuadTree;
import org.geotools.data.shapefile.index.quadtree.StoreException;
import org.geotools.data.shapefile.index.quadtree.fs.FileSystemIndexStore;
import org.geotools.data.shapefile.shp.IndexFile;
import org.geotools.data.shapefile.shp.ShapefileHeader;
import org.geotools.data.shapefile.shp.ShapefileReader;
import org.geotools.data.util.NullProgressListener;
import org.geotools.util.logging.Logging;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.GeometryFactory;
import org.opengis.util.ProgressListener;

class ShapeFileIndexer
implements FileWriter {
    private static final Logger LOGGER = Logging.getLogger(ShapeFileIndexer.class);
    private int maxDepth = -1;
    private int leafSize = 16;
    private String byteOrder;
    private ShpFiles shpFiles;

    ShapeFileIndexer() {
    }

    public static void main(String[] args) throws IOException {
        if (args.length < 1 || (args.length - 1) % 2 != 0) {
            ShapeFileIndexer.usage();
        }
        long start = System.currentTimeMillis();
        ShapeFileIndexer idx = new ShapeFileIndexer();
        for (int i = 0; i < args.length; ++i) {
            if (args[i].equals("-t")) {
                ++i;
                continue;
            }
            if (args[i].equals("-M")) {
                idx.setMax(Integer.parseInt(args[++i]));
                continue;
            }
            if (args[i].equals("-s")) {
                idx.setLeafSize(Integer.parseInt(args[++i]));
                continue;
            }
            if (args[i].equals("-b")) {
                idx.setByteOrder(args[++i]);
                continue;
            }
            if (!args[i].toLowerCase().endsWith(".shp")) {
                System.out.println("File extension must be '.shp'");
                System.exit(1);
            }
            idx.setShapeFileName(new ShpFiles(args[i]));
        }
        try {
            System.out.print("Indexing ");
            int cnt = idx.index(true, (ProgressListener)new NullProgressListener());
            System.out.println();
            System.out.print(cnt + " features indexed ");
            System.out.println("in " + (System.currentTimeMillis() - start) + "ms.");
            System.out.println();
        }
        catch (Exception e) {
            Logger.getGlobal().log(Level.INFO, "", e);
            ShapeFileIndexer.usage();
            System.exit(1);
        }
    }

    private static void usage() {
        System.out.println("Usage: ShapeFileIndexer -t <QIX> [-M <max tree depth>] [-b <byte order NL | NM>] <shape file>[-s <max number of items in a leaf>]");
        System.out.println();
        System.out.println("Options:");
        System.out.println("\t-t Index type: RTREE or QUADTREE");
        System.out.println();
        System.out.println("Following options apllies only to QUADTREE:");
        System.out.println("\t-b byte order to use: NL = LSB; NM = MSB (default)");
        System.exit(1);
    }

    public int index(boolean verbose, ProgressListener listener) throws MalformedURLException, IOException, TreeException, StoreException, LockTimeoutException {
        if (this.shpFiles == null) {
            throw new IOException("You have to set a shape file name!");
        }
        int cnt = 0;
        StorageFile storage = this.shpFiles.getStorageFile(ShpFileType.QIX);
        File treeFile = storage.getFile();
        if (this.maxDepth == -1) {
            this.maxDepth = this.computeMaxDepth();
        }
        try (ShapefileReader reader = new ShapefileReader(this.shpFiles, true, false, new GeometryFactory());){
            cnt = this.buildQuadTree(reader, treeFile, verbose);
        }
        storage.replaceOriginal();
        return cnt;
    }

    private int computeMaxDepth() throws IOException {
        try (ShapefileReader reader = new ShapefileReader(this.shpFiles, true, false, new GeometryFactory());){
            int features = reader.getCount(0);
            int maxDepth = 1;
            int nodes = 1;
            while (nodes * this.leafSize < features) {
                ++maxDepth;
                nodes *= 4;
            }
            if (maxDepth < 10) {
                maxDepth = 10;
            }
            int n = maxDepth;
            return n;
        }
    }

    private int buildQuadTree(ShapefileReader reader, File file, boolean verbose) throws IOException, StoreException {
        LOGGER.fine("Building quadtree spatial index with depth " + this.maxDepth + " for file " + file.getAbsolutePath());
        byte fileByteOrder = this.resolveStorageByteOrder();
        int cnt = 0;
        try (IndexFile shpIndex = new IndexFile(this.shpFiles, false);
             ShapefileIndexerBoundsHelper.BoundsReader boundsHelper = ShapefileIndexerBoundsHelper.createBoundsReader(reader, shpIndex);
             QuadTree tree = new QuadTree(shpIndex.getRecordCount(), this.maxDepth, this.getBounds(reader), shpIndex);){
            Envelope env = new Envelope();
            while (reader.hasNext()) {
                ShapefileReader.Record rec = reader.nextRecord();
                env.init(rec.minX, rec.maxX, rec.minY, rec.maxY);
                int recno = cnt++;
                tree.insert(recno, env);
                boundsHelper.insert(recno, env);
                if (verbose && cnt % 1000 == 0) {
                    System.out.print('.');
                }
                if (cnt % 100000 != 0) continue;
                System.out.print('\n');
            }
            if (verbose) {
                System.out.println("done building quadtree");
            }
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine("Optimizing the tree (this might take some time)");
            }
            if (verbose) {
                System.out.println("Optimizing the tree (this might take some time)");
            }
            this.optimizeTree(tree, tree.getRoot(), 0, boundsHelper);
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine("Tree optimized");
            }
            if (LOGGER.isLoggable(Level.FINE)) {
                this.printStats(tree);
            }
            if (verbose) {
                System.out.println("Storing the tree...");
            }
            FileSystemIndexStore store = new FileSystemIndexStore(file, fileByteOrder);
            store.store(tree);
            if (verbose) {
                System.out.println("done");
            }
        }
        return cnt;
    }

    private Envelope getBounds(ShapefileReader reader) {
        ShapefileHeader header = reader.getHeader();
        Envelope bounds = new Envelope(header.minX(), header.maxX(), header.minY(), header.maxY());
        return bounds;
    }

    private byte resolveStorageByteOrder() throws StoreException {
        if (this.byteOrder == null || this.byteOrder.equalsIgnoreCase("NM")) {
            return 2;
        }
        if (this.byteOrder.equalsIgnoreCase("NL")) {
            return 1;
        }
        throw new StoreException("Asked byte order '" + this.byteOrder + "' must be 'NL' or 'NM'!");
    }

    private Node optimizeTree(QuadTree tree, Node node, int level, ShapefileIndexerBoundsHelper.BoundsReader reader) throws StoreException, IOException {
        int i;
        int i2;
        boolean canBeSplit;
        boolean isLeafNode = node.getNumSubNodes() == 0;
        boolean isOverFlown = node.getNumShapeIds() > this.leafSize;
        int hardMaxDepth = this.maxDepth * 2;
        boolean bl = canBeSplit = level < hardMaxDepth;
        if (isLeafNode && isOverFlown && canBeSplit) {
            int[] shapeIds = node.getShapesId();
            int numShapesId = node.getNumShapeIds();
            node.clean();
            int extraLevels = 2;
            int nodes = 4;
            while (nodes * this.leafSize < numShapesId) {
                ++extraLevels;
                nodes *= 4;
            }
            Envelope env = new Envelope();
            for (int i3 = 0; i3 < numShapesId; ++i3) {
                int recNumber = shapeIds[i3];
                reader.read(recNumber, env);
                tree.insert(node, recNumber, env, extraLevels);
            }
        }
        node.pack();
        for (i2 = 0; i2 < node.getNumSubNodes(); ++i2) {
            this.optimizeTree(tree, node.getSubNode(i2), level + 1, reader);
        }
        i2 = 0;
        while (i2 < node.getNumSubNodes()) {
            Node child = node.getSubNode(i2);
            if (child != null && child.getNumShapeIds() == 0 && child.getNumSubNodes() == 0) {
                node.removeSubNode(child);
                continue;
            }
            ++i2;
        }
        if (node.getNumSubNodes() == 1 && node.getNumShapeIds() == 0) {
            Node subnode = node.getSubNode(0);
            node.clearSubNodes();
            node.setShapesId(subnode);
            node.setBounds(subnode.getBounds());
            for (i = 0; i < subnode.getNumSubNodes(); ++i) {
                node.addSubNode(subnode.getSubNode(i));
            }
        } else {
            int i4;
            Envelope bounds = new Envelope();
            if (node.getNumShapeIds() > 0) {
                int[] shapeIds;
                for (int recNumber : shapeIds = node.getShapesId()) {
                    reader.expandEnvelope(recNumber, bounds);
                }
            }
            if (node.getNumSubNodes() > 0) {
                for (i = 0; i < node.getNumSubNodes(); ++i) {
                    bounds.expandToInclude(node.getSubNode(i).getBounds());
                }
            }
            node.setBounds(bounds);
            int count = node.getNumShapeIds();
            for (i4 = 0; i4 < node.getNumSubNodes(); ++i4) {
                Node child = node.getSubNode(i4);
                if (child.getNumSubNodes() > 0) {
                    count = Integer.MAX_VALUE;
                    break;
                }
                count += child.getNumShapeIds();
            }
            if (count < this.leafSize) {
                for (i4 = 0; i4 < node.getNumSubNodes(); ++i4) {
                    Node child = node.getSubNode(i4);
                    int[] shapesId = child.getShapesId();
                    for (int j = 0; j < child.getNumShapeIds(); ++j) {
                        node.addShapeId(shapesId[j]);
                    }
                }
                node.clearSubNodes();
            }
        }
        return node;
    }

    private void printStats(QuadTree tree) throws StoreException {
        HashMap<Integer, Integer> stats = new HashMap<Integer, Integer>();
        this.gatherStats(tree.getRoot(), stats);
        ArrayList nums = new ArrayList(stats.keySet());
        Collections.sort(nums);
        LOGGER.log(Level.FINE, "Index statistics");
        for (Integer num : nums) {
            LOGGER.log(Level.FINE, num + " -> " + String.valueOf(stats.get(num)));
        }
    }

    void gatherStats(Node node, Map<Integer, Integer> stats) throws StoreException {
        int num = node.getNumShapeIds();
        Integer count = stats.get(num);
        if (count == null) {
            stats.put(num, 1);
        } else {
            stats.put(num, count + 1);
        }
        for (int i = 0; i < node.getNumSubNodes(); ++i) {
            this.gatherStats(node.getSubNode(i), stats);
        }
    }

    public void setMax(int i) {
        this.maxDepth = i;
    }

    public void setShapeFileName(ShpFiles shpFiles) {
        this.shpFiles = shpFiles;
    }

    public void setByteOrder(String byteOrder) {
        this.byteOrder = byteOrder;
    }

    @Override
    public String id() {
        return this.getClass().getName();
    }

    public int getLeafSize() {
        return this.leafSize;
    }

    public void setLeafSize(int leafSize) {
        if (leafSize < 1) {
            throw new IllegalArgumentException("Maximum node leaf size must be a positive integer");
        }
        this.leafSize = leafSize;
    }
}

